C      *****************************************************************
C      * Subroutine PropDecay                                          *
C      * Updates KTs, for the passed layer, for phosphorus and sulfur  *
C      * assuming that the decay is proportional to the nitrogen       *
C      * mineralization. If an element is not present then the decay   *
C      * calculations for that element are skipped.                    *
C      *                                                               *
C      * Variables:                                                    *
C      * GPPerMole -Local - The number of grams of phosphorus per      *
C      *                    mole. Units are g P/mol P.                 *
C      * GSPerMole -Local - The number of grams of sulfur per  mole.   *
C      *                    Units are g S/mol S.                       *
C      * KT        -In/Out- A matrix, of size NComp by NLayer, of the  *
C      *                    known total amount for each component, for *
C      *                    each layer.  The total includes the amount *
C      *                    of the component as a free component and as*
C      *                    part of all dependent species.  Units are  *
C      *                    mol.                                       *
C      *                    (Common block VModel, file VModel.f)       *
C      * LfNut     -Input - A vector, of size NComp, of the amount of  *
C      *                    each nutrient in the leaves that senesced  *
C      *                    during the day.  Units are g nutrients.    *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * Lyr       -Input - Layer array index.  Unitless.              *
C      * Name      -Local - Used for passing data to WhereIs function. *
C      *                    Unitless.                                  *
C      * Net_N_Min -Input - The net nitrogen mineralized.  Units are   *
C      *                    g N.                                       *
C      * N_in_OM   -Input - The amount of nitrogen in the organic      *
C      *                    matter.  Units are g N.                    *
C      * NLayers   -Input - The number of soil layers in the system.   *
C      *                    Input from TREGRO, or read into the model  *
C      *                    by the routine InParam.  Unitless.         *
C      *                    (Common block VPos, file VModel.f)         *
C      * P_in_OM   -In/Out- The amount of phosphorus in organic        *
C      *                    matter.  Units are g P.                    *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * P_Min     -Local - The amount of phosphorus mineralized.      *
C      *                    Units are g P.                             *
C      * PNum      -Local - Index for calcium.  Unitless.              *
C      * P_Prop    -Local - The proportion between phosphorus and      *
C      *                    nitrogen in orgranic matter.  Unitless.    *
C      * RtNut     -Input - A matrix, NLayers by NComps, of the        *
C      *                    amount of nutrients in the roots, in each  *
C      *                    layer, that senesced during the day.  Input*
C      *                    from TREGRO, or read into the model by the *
C      *                    routine InParam.  Units are g nutr.        *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * S_in_OM   -In/Out- The amount of sulfur in organic matter.    *
C      *                    Units are g S.                             *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * SNum      -Local - Index for potassium.  Unitless.            *
C      * S_Min     -Local - The amount of sulfur mineralized.  Units   *
C      *                    are g S.                                   *
C      * S_Prop    -Local - The proportion between sulfur and          *
C      *                    nitrogen in orgranic matter.  Unitless.    *
C      * WhereIs   -Fcn   - Returns the index of requested species. If *
C      *                    the species is not found a -1 is returned. *
C      *                    Unitless.                                  *
C      *****************************************************************
       SUBROUTINE PROPDECAY(LYR, N_IN_OM, NET_N_MIN)
       IMPLICIT NONE
       INCLUDE 'VMODEL.INC'
       INTEGER PNUM, LYR, SNUM, WHEREIS
       REAL*8 GPPERMOLE, GSPERMOLE
       REAL*8 MINP, N_IN_OM, NET_N_MIN, P_MIN, P_PROP, S_MIN, S_PROP
							CHARACTER*8 NAME
       PARAMETER (GPPERMOLE = 31.0, GSPERMOLE = 32.0)

C      *------------------*
C      * Get the indices. *
C      *------------------*
       NAME = 'P'
       PNUM = WHEREIS(NAME)
       NAME = 'S'
       SNUM = WHEREIS(NAME)

C      *---------------------------*
C      * Update KT for phosphorus. *
C      *---------------------------*
       IF (PNUM.NE.-1) THEN

C         *-----------------------------------*
C         * Calculate the amount mineralized. *
C         *-----------------------------------*
          P_PROP = P_IN_OM(LYR) / N_IN_OM
          P_MIN = P_PROP * NET_N_MIN

C         *-------------------------------*
C         * Update the KT for phosphorus. *
C         *-------------------------------*
          KT(PNUM,LYR) = KT(PNUM,LYR) + P_MIN / GPPERMOLE

C         *-------------------------------------------------*
C         * Update the total organic matter for phosphorus. *
C         *-------------------------------------------------*
          P_IN_OM(LYR) = MAX(0.0, P_IN_OM(LYR) - P_MIN +
     >                   RTNUT(PNUM,LYR) + LFNUT(PNUM))
     
       ENDIF

C      *-----------------------*
C      * Update KT for sulfur. *
C      *-----------------------*
       IF (SNUM.NE.-1) THEN

C         *-----------------------------------*
C         * Calculate the amount mineralized. *
C         *-----------------------------------*
          S_PROP = S_IN_OM(LYR) / N_IN_OM
          S_MIN = S_PROP * NET_N_MIN

C         *---------------------------------------------------------*
C         * Update the KT for sulfur. Need to convert from grams to *
C         * moles.                                                  *
C         *---------------------------------------------------------*
          KT(SNUM,LYR) = KT(SNUM,LYR) + S_MIN / GSPERMOLE

C         *---------------------------------------------*
C         * Update the total organic matter for sulfur. *
C         *---------------------------------------------*
          S_IN_OM(LYR) = MAX(0.0, S_IN_OM(LYR) - S_MIN +
     >                   RTNUT(SNUM,LYR) + LFNUT(SNUM))
       ENDIF

	      RETURN
	      END
C      *****************************************************************
C      *                    END SUBROUTINE                             *
C      *****************************************************************
